//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.2

import wesual.Controls 1.0
import wesual.Ui       1.0

import wesual.data.Core 1.0

GridView {
    id : gridView

    readonly property alias dragDocument : p_.dragDocument
    readonly property alias containsDrag : dropZone.containsDrag
    readonly property int   cellsPerRow  :
        Math.max(1, Math.floor(width / cellWidth))

    property int dropIndicatorHeight : 0

    signal documentDropped(Document dropped)

    interactive : false
    clip : dragDocument === null ? true : false
    boundsBehavior : Flickable.StopAtBounds

    onFlowChanged : {
        if (flow !== GridView.FlowLeftToRight) {
            throw new Error("DragGridView does " +
                            "only support left to right flow");
        }
    }

    QtObject {
        id : p_

        property Document dragDocument
    }

    move : Transition {
        SmoothedAnimation {
            properties : "x,y"
            duration: 200
        }
        NumberAnimation {
            duration : 150
            property : "opacity"
            to : 1
        }
    }
    displaced : move
    remove : Transition {
        NumberAnimation {
            duration : 150
            properties : "opacity,scale"
            from : 1
            to : 0.01
        }
    }
    add : Transition {
        SequentialAnimation {
            PropertyAction {
                property : "opacity"
                value : 0
            }
            PauseAnimation { duration : 100 }
            NumberAnimation {
                duration : 150
                property : "opacity"
                from : 0
                to : 1
            }
            alwaysRunToEnd : true
        }
    }
    populate : add

    NumberAnimation {
        id : scrollAnimation
        target : gridView
        property : "contentY"
        easing.type : Easing.Linear
    }

    Item {
        id : dropIndicator

        z : 100
        visible : false
        opacity : 0

        Behavior on x {
            enabled : dropIndicator.visible
            SmoothedAnimation {
                duration: 200
            }
        }
        Behavior on y {
            enabled : dropIndicator.visible
            SmoothedAnimation {
                duration: 200
            }
        }

        states : State {
            name : "visible"
            when : dropIndicator.visible

            PropertyChanges {
                target : dropIndicator
                opacity : 1
            }
        }

        transitions : [
            Transition {
                to : "visible"

                SequentialAnimation {
                    PropertyAction {
                        target : dropIndicator
                        property : "opacity"
                        value : 0
                    }
                    PropertyAction {
                        target : dropIndicator
                        property : "visible"
                        value : true
                    }
                    NumberAnimation {
                        target : dropIndicator
                        duration : 200
                        property : "opacity"
                    }
                }
            },
            Transition {
                from : "visible"

                SequentialAnimation {
                    NumberAnimation {
                        target : dropIndicator
                        duration : 200
                        property : "opacity"
                    }
                    PropertyAction {
                        target : dropIndicator
                        property : "visible"
                        value : false
                    }
                }
            }
        ]

        Rectangle {
            width : 3
            height : (gridView.dropIndicatorHeight === 0) ? gridView.cellHeight
                                                : gridView.dropIndicatorHeight
            anchors {
                horizontalCenter : parent.horizontalCenter
                top : parent.top
            }
            color : UI.color(UI.SecondaryHover)
        }
    }

    UiDropZone {
        id : dropZone

        z : 1
        anchors.fill : parent

        DraggableDropHandler {
            property bool removeOnLeave

            key : "we.captivo.DragDelegate"

            function contentPosition(viewPos) {
                return dropZone.mapToItem(
                            gridView.contentItem, viewPos.x, viewPos.y);
            }

            function rowForContentPosition(contentPos) {
                return Math.floor((contentPos.y - originY) / cellHeight);
            }

            function columnForContentPosition(contentPos) {
                return Math.floor((contentPos.x - originX) / cellWidth);
            }

            function cellIndex(contentPos) {
                var row = rowForContentPosition(contentPos)
                var col = columnForContentPosition(contentPos);

                var index = row * gridView.cellsPerRow + col;

                var cellPos = (contentPos.x - originX) - col * cellWidth;
                if (cellPos > cellWidth / 2) {
                    index += 1;
                }

                return Math.min(index, gridView.count);
            }

            function cellPosition(index) {
                var row = Math.floor(index / gridView.cellsPerRow);
                var col = index % gridView.cellsPerRow;

                return Qt.point(col * cellWidth + originX,
                                row * cellHeight + originY);
            }

            function positionIndicator(cellIndex, contentPos) {
                var pos = cellPosition(cellIndex);
                pos = gridView.mapFromItem(
                            gridView.contentItem, pos.x, pos.y);

                if (cellIndex % gridView.cellsPerRow === 0) {
                    // Corner case: Position indicator according to row
                    var visualRow = rowForContentPosition(contentPos);
                    var row = Math.floor(cellIndex / gridView.cellsPerRow);
                    if (visualRow !== row) {
                        pos = cellPosition(cellIndex - 1);
                        pos.x += gridView.cellWidth;
                    }
                }

                dropIndicator.x = pos.x;
                dropIndicator.y = pos.y;
                dropIndicator.visible = true;
            }

            function autoScroll(pos) {
                return false;

                /*var from, to, scroll = false;

                var scrollHorizontal =
                        gridView.orientation === Qt.Horizontal &&
                        gridView.visibleArea.widthRatio < 1;
                var scrollVertical =
                        gridView.orientation === Qt.Vertical &&
                        gridView.visibleArea.heightRatio < 1
                var margin = gridView.autoScrollMargin;

                if (scrollHorizontal) {
                    if (pos.x < gridView.width && pos.x > gridView.width - margin) {
                        from = gridView.contentX;
                        to = p_.maxContentX;
                        scroll = true;
                    } else if (pos.x > 0 && pos.x < margin) {
                        from = gridView.contentX;
                        to = gridView.originX - gridView.leftMargin;
                        scroll = true;
                    }
                } else if (scrollVertical) {
                    if (pos.y < gridView.height && pos.y > gridView.height - margin) {
                        from = gridView.contentY;
                        to = p_.maxContentY;
                        scroll = true;
                    } else if (pos.y > 0 && pos.y < margin) {
                        from = gridView.contentY;
                        to = gridView.originY - gridView.topMargin;
                        scroll = true;
                    }
                }

                if (scroll && !scrollAnimation.running) {
                    scrollAnimation.from = from;
                    scrollAnimation.to = to;
                    scrollAnimation.duration = Math.abs(to - from);
                    scrollAnimation.start();
                }

                return scroll;*/
            }

            onEntered : {
                removeOnLeave = false;
                if (model && model.isModifyable) {
                    var doc = drag.source.document;
                    var currentIndex = model.indexOf(doc);

                    if (currentIndex !== -1) {
                        drag.source.originatingIndex = currentIndex;
                    }
                    p_.dragDocument = doc;
                }
            }

            onPositionChanged : {
                if (!model || !model.isModifyable)
                    return;

                // Autoscroll
                if (autoScroll(drag.position))
                    return;

                scrollAnimation.stop();

                var pos = contentPosition(drag.position);
                var index = cellIndex(pos);

                // Position indicator
                positionIndicator(index, pos);
            }

            onExited : {
                if (!model.isModifyable)
                    return;

                scrollAnimation.stop();
                dropIndicator.visible = false;

                p_.dragDocument = null;
            }

            onDropped : {
                if (!model.isModifyable)
                    return;

                scrollAnimation.stop();
                var doc = p_.dragDocument;
                p_.dragDocument = null;

                var pos = contentPosition(drag.position);
                var index = cellIndex(pos);
                dropIndicator.visible = false;

                var drop = true;
                var currentIndex = model.indexOf(doc);
                if (currentIndex !== -1) {
                    if (currentIndex < index) {
                        index = Math.max(0, index - 1);
                    }

                    if (index === currentIndex)
                        drop = false;
                }

                if (drop) {
                    model.insertElement(drag.source.document, index);

                    gridView.documentDropped(doc);
                }
            }
        }
    }
}
